import { onMounted, onUnmounted, Ref, ref } from "vue";
import debounce from "lodash/debounce";
import * as Type from "./overview/type";
import * as Event from "./overview/event";
import * as Api from "./overview/api";
import * as Data from "./overview/data";
import * as Validation from "./overview/validation";
import { MENU_TOGGLED } from "layout/events";
import { Widget } from "./overview/type";
import _ from "lodash";

const EVENT_RESIZE = "resize";
const REFRESH_INTERVAL_MULTIPLIER = 1000;

export const lastAddedWidget: Ref<Widget | null> = ref(null);

const handlers = {
    async getWidgets(): Promise<void> {
        Data.isLoading.value = true;

        let widgetsBeforeRefresh = null;

        if (Data.widgets.value.length) {
            widgetsBeforeRefresh = JSON.parse(JSON.stringify(Data.widgets.value));
        }

        try {
            const widgets: Type.Widgets = await Api.getWidgets();

            await Validation.validateSchema(Validation.Schema.GetWidgetsResponseBody, widgets);

            if (widgetsBeforeRefresh !== null) {
                lastAddedWidget.value = _.differenceWith(widgets, widgetsBeforeRefresh, _.isEqual)[0];
            }

            Data.widgets.value = widgets;
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }

        Data.isLoading.value = false;
    },
    async getWidget(id: Type.Id): Promise<void> {
        Data.isLoading.value = true;

        const index = Data.widgets.value.findIndex((widget) => widget.id === id);

        Data.widgetLoading.value = Data.widgets.value[index];

        try {
            const widget: Type.Widget = await Api.getWidget(id);

            await Validation.validateSchema(Validation.Schema.GetWidgetResponseBody, widget);
            Data.widgets.value[index] = widget;
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }

        Data.widgetLoading.value = Data.UNSELECTED_WIDGET;
        Data.isLoading.value = false;
    },
    async getAvailableWidgets(): Promise<void> {
        Data.isLoading.value = true;

        try {
            const availableWidgets: Type.AvailableWidgets = await Api.getAvailableWidgets();

            await Validation.validateSchema(Validation.Schema.GetAvailableWidgetsResponseBody, availableWidgets);
            Data.availableWidgets.value = availableWidgets;
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }

        Data.isLoading.value = false;
    },
    async addWidget(type: Type.Type): Promise<void> {
        Data.isLoading.value = true;

        try {
            await Api.addWidget(type);
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }

        Event.invokeEventHandler(Event.AddWidgetModalEvent.Close);
        Event.invokeEventHandler(Event.WidgetEvent.GetWidgets);

        Data.isLocked.value = false;
        Data.isLoading.value = false;
    },
    async deleteWidget(id: Type.Id): Promise<void> {
        Data.isLoading.value = true;

        const index = Data.widgets.value.findIndex((widget) => widget.id === id);

        Data.widgetLoading.value = Data.widgets.value[index];

        try {
            await Api.deleteWidget(id);
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }

        Event.invokeEventHandler(Event.DeleteWidgetModalEvent.Close);

        Data.widgets.value.splice(index, 1);

        Data.widgetLoading.value = Data.UNSELECTED_WIDGET;
        Data.isLoading.value = false;
    },
    async configureWidget(id: Type.Id, optionUpdateSet: Type.OptionUpdateSet): Promise<void> {
        Data.isLoading.value = true;

        try {
            await Api.configureWidget(id, optionUpdateSet);
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }
        Event.invokeEventHandler(Event.EditWidgetModalEvent.Close);
        Event.invokeEventHandler(Event.WidgetEvent.GetWidget, id);

        Data.isLoading.value = false;
    },
    async renameWidget(id: Type.Id, label: Type.Label): Promise<void> {
        Data.isLoading.value = true;

        try {
            await Api.renameWidget(id, label);
        } catch (error) {
            if (error instanceof Error) {
                handlers.showError(error);
            }
        }

        Event.invokeEventHandler(Event.EditWidgetModalEvent.Close);
        Event.invokeEventHandler(Event.WidgetEvent.GetWidget, id);

        Data.isLoading.value = false;
    },
    openEditWidgetModal(widget: Type.Widget): void {
        Data.widgetBeingEdited.value = widget;
        Data.isEditWidgetModalOpen.value = true;
    },
    closeEditWidgetModal(): void {
        Data.isEditWidgetModalOpen.value = false;
    },
    openAddWidgetModal(): void {
        Data.isAddWidgetModalOpen.value = true;
    },
    closeAddWidgetModal(): void {
        Data.isAddWidgetModalOpen.value = false;
    },
    openDeleteWidgetModal(widget: Type.Widget): void {
        Data.widgetBeingDeleted.value = widget;
        Data.isDeleteWidgetModalOpen.value = true;
    },
    closeDeleteWidgetModal(): void {
        Data.isDeleteWidgetModalOpen.value = false;
    },
    closeErrorModal(): void {
        Data.isErrorModalOpen.value = false;
        Data.errorMessageBeingShown.value = Data.NO_ERROR_MESSAGE;
    },
    toggleLock(): void {
        Data.isLocked.value = !Data.isLocked.value;
    },
    refresh(): void {
        Event.invokeEventHandler(Event.WidgetEvent.GetWidgets);
    },
    updateRefreshInterval(interval: Type.RefreshInterval | Type.UnselectedRefreshInterval): void {
        Data.refreshInterval.value = interval;

        window.clearInterval(Data.refreshIntervalLoop.value);

        if (interval === Data.UNSELECTED_REFRESH_INTERVAL) {
            return;
        }

        Data.refreshIntervalLoop.value = window.setInterval(
            () => Event.invokeEventHandler(Event.WidgetEvent.GetWidgets),
            interval * REFRESH_INTERVAL_MULTIPLIER
        );
    },
    showError(error: Error): void {
        Data.isErrorModalOpen.value = true;
        Data.errorMessageBeingShown.value = error.message;
    },
    updateStyle: debounce(() => {
        const container = document.getElementById("gx-main");

        if (!container) {
            return;
        }

        Data.pageStyle.value = {
            width: `${container.offsetWidth}px`,
            height: `${container.offsetHeight - Data.titleBarHeight.value}px`,
        };

        Data.pageKey.value = Date.now();
    }, 500),
    validateScreenSize(): void {
        Data.isSmallScreen.value = window.innerWidth <= 1024;
    },
};

const events: [Event.Event, Function][] = [
    [Event.WidgetEvent.GetWidgets, handlers.getWidgets],
    [Event.WidgetEvent.GetWidget, handlers.getWidget],
    [Event.WidgetEvent.GetAvailableWidgets, handlers.getAvailableWidgets],
    [Event.WidgetEvent.Add, handlers.addWidget],
    [Event.WidgetEvent.Delete, handlers.deleteWidget],
    [Event.WidgetEvent.Configure, handlers.configureWidget],
    [Event.WidgetEvent.Resize, Api.resizeWidget],
    [Event.WidgetEvent.Move, Api.moveWidget],
    [Event.WidgetEvent.Rename, handlers.renameWidget],
    [Event.EditWidgetModalEvent.Open, handlers.openEditWidgetModal],
    [Event.EditWidgetModalEvent.Close, handlers.closeEditWidgetModal],
    [Event.AddWidgetModalEvent.Open, handlers.openAddWidgetModal],
    [Event.AddWidgetModalEvent.Close, handlers.closeAddWidgetModal],
    [Event.DeleteWidgetModalEvent.Open, handlers.openDeleteWidgetModal],
    [Event.DeleteWidgetModalEvent.Close, handlers.closeDeleteWidgetModal],
    [Event.ErrorModalEvent.Close, handlers.closeErrorModal],
    [Event.LockToggleButtonEvent.ToggleLock, handlers.toggleLock],
    [Event.RefreshButtonEvent.Refresh, handlers.refresh],
    [Event.RefreshButtonEvent.UpdateRefreshInterval, handlers.updateRefreshInterval],
];

const eventsAtBoot: Event.Event[] = [Event.WidgetEvent.GetWidgets, Event.WidgetEvent.GetAvailableWidgets];

export async function boot(): Promise<void> {
    handlers.updateStyle();

    onMounted(() => {
        events.forEach(([event, callback]) => Event.registerEventHandler(event as Event.Event, callback as Function));
        window.addEventListener(EVENT_RESIZE, handlers.updateStyle);
        window.addEventListener(EVENT_RESIZE, handlers.validateScreenSize);
        window.addEventListener(MENU_TOGGLED, handlers.updateStyle);
        eventsAtBoot.forEach(Event.invokeEventHandler);
    });

    onUnmounted(() => {
        window.clearInterval(Data.refreshIntervalLoop.value);
        window.removeEventListener(EVENT_RESIZE, handlers.updateStyle);
        window.removeEventListener(EVENT_RESIZE, handlers.validateScreenSize);
        window.removeEventListener(MENU_TOGGLED, handlers.updateStyle);
        Event.unregisterEventHandlers(...(events.map(([event]: [Event.Event, Function]) => event) as Event.Event[]));
    });
}

export default {
    ...Data,
    ...Event,
};
